#include <complex.h>
#include <math.h>
#include <stddef.h>

#include "scf.h"
#include "scf_code.h"
#include "scf_filter.h"
#include "scf_tx.h"

static complex float fir_tail[SCF_FIR_LEN_TX];
static float carrier_freq;
static float carrier_phase;
static const float freq_step = (float) SCF_SRATE / (float) SCF_CHIP_LEN;

void scf_tx_init(float freq)
{
    scf_filter_init();

    carrier_freq = freq;

    complex float initial_freq[SCF_CHIP_LEN];
    for (size_t i = 0; i < SCF_CHIP_LEN; i++) {
        initial_freq[i] = carrier_freq;
    }
    complex float initial_freq_filtered[SCF_CHIP_LEN];
    scf_filter_tx(initial_freq_filtered, initial_freq, fir_tail);
}

void scf_tx(float *passband, uint8_t symbol)
{
    size_t passband_idx = 0;
    for (size_t c = 0; c < SCF_CHIPS; c++) {
        float freq = carrier_freq + freq_step * (float) scf_code[symbol][c];
        complex float modulation[SCF_CHIP_LEN];
        for (size_t i = 0; i < SCF_CHIP_LEN; i++) {
            modulation[i] = freq;
        }

        complex float modulation_filtered[SCF_CHIP_LEN];
        scf_filter_tx(modulation_filtered, modulation, fir_tail);

        for (size_t i = 0; i < SCF_CHIP_LEN; i++) {
            float freq = crealf(modulation_filtered[i]);
            passband[passband_idx] = sinf(carrier_phase);
            passband_idx++;

            carrier_phase += 2.0f * M_PI * freq * (1.0f / (float) SCF_SRATE);
            while (carrier_phase > 2.0f * M_PI) {
                carrier_phase -= 2.0f * M_PI;
            }
        }
    }
}
