#include <complex.h>
#include <fftw3.h>
#include <assert.h>
#include <string.h>

#include "scf.h"
#include "scf_filter.h"

#define FIR_FFT_LEN 512

static const complex float bb_fir_rx_td[];
static const complex float bb_fir_tx_td[];
static complex float bb_fir_rx_fd[FIR_FFT_LEN];
static complex float bb_fir_tx_fd[FIR_FFT_LEN];

static fftwf_complex *fft_in;
static fftwf_complex *fft_out;
static fftwf_plan fft;

static fftwf_complex *ifft_in;
static fftwf_complex *ifft_out;
static fftwf_plan ifft;

static unsigned int initialized;

static void scf_filter_internal(complex float *out, complex float *in, complex float *tail, size_t tail_len, complex float *fir_fd)
{
    assert(initialized);
    assert(SCF_CHIP_LEN + tail_len <= FIR_FFT_LEN);

    memset(fft_in, 0, FIR_FFT_LEN * sizeof(fft_in[0]));
    memcpy(fft_in, in, SCF_CHIP_LEN * sizeof(fft_in[0]));

    fftwf_execute(fft);

    memcpy(ifft_in, fft_out, FIR_FFT_LEN * sizeof(ifft_in[0]));
    for (size_t i = 0; i < FIR_FFT_LEN; i++) {
        ifft_in[i] *= fir_fd[i];
    }

    fftwf_execute(ifft);

    for (size_t i = 0; i < FIR_FFT_LEN; i++) {
        ifft_out[i] /= (float) FIR_FFT_LEN;
    }
    for (size_t i = 0; i < tail_len; i++) {
        ifft_out[i] += tail[i];
    }
    memcpy(out, ifft_out, SCF_CHIP_LEN * sizeof(out[0]));
    memcpy(tail, &ifft_out[SCF_CHIP_LEN], tail_len * sizeof(tail[0]));
}

void scf_filter_init(void)
{
    if (initialized)
        return;

    fft_in = fftwf_alloc_complex(FIR_FFT_LEN);
    fft_out = fftwf_alloc_complex(FIR_FFT_LEN);
    assert(fft_in);
    assert(fft_out);
    fft = fftwf_plan_dft_1d(FIR_FFT_LEN, fft_in, fft_out, FFTW_FORWARD, FFTW_ESTIMATE);
    assert(fft);

    ifft_in = fftwf_alloc_complex(FIR_FFT_LEN);
    ifft_out = fftwf_alloc_complex(FIR_FFT_LEN);
    assert(ifft_in);
    assert(ifft_out);
    ifft = fftwf_plan_dft_1d(FIR_FFT_LEN, ifft_in, ifft_out, FFTW_BACKWARD, FFTW_ESTIMATE);
    assert(ifft);

    memset(fft_in, 0, sizeof(fft_in[0]) * FIR_FFT_LEN);
    memcpy(fft_in, bb_fir_rx_td, SCF_FIR_LEN_RX * sizeof(fft_in[0]));
    fftwf_execute(fft);
    memcpy(bb_fir_rx_fd, fft_out, sizeof(bb_fir_rx_fd));

    memset(fft_in, 0, sizeof(fft_in[0]) * FIR_FFT_LEN);
    memcpy(fft_in, bb_fir_tx_td, SCF_FIR_LEN_TX * sizeof(fft_in[0]));
    fftwf_execute(fft);
    memcpy(bb_fir_tx_fd, fft_out, sizeof(bb_fir_tx_fd));

    initialized = 1;
}

void scf_filter_rx(complex float *out, complex float *in, complex float *tail)
{
    scf_filter_internal(out, in, tail, SCF_FIR_LEN_RX, bb_fir_rx_fd);
}

void scf_filter_tx(complex float *out, complex float *in, complex float *tail)
{
    scf_filter_internal(out, in, tail, SCF_FIR_LEN_TX, bb_fir_tx_fd);
}

// https://fiiir.com/
// Low pass 1000 Hz
static const complex float bb_fir_rx_td[SCF_FIR_LEN_RX] = {
    -0.000233771000961963,
    0.000000000000000002,
    0.000244596824120296,
    0.000358803521938844,
    0.000265419919503219,
    -0.000000000000000001,
    -0.000296988501407617,
    -0.000448364890214722,
    -0.000340071984727431,
    -0.000000000000000001,
    0.000395465279877511,
    0.000605332650800241,
    0.000463994335931762,
    -0.000000000000000001,
    -0.000546523198228378,
    -0.000839081381695142,
    -0.000643962913379392,
    0.000000000000000005,
    0.000757282701751799,
    0.001159965940927727,
    0.000887523932257827,
    -0.000000000000000002,
    -0.001035817586776309,
    -0.001579842403752117,
    -0.001203406105977533,
    -0.000000000000000004,
    0.001391670785141685,
    0.002112884021067343,
    0.001602166267302543,
    -0.000000000000000004,
    -0.001836664205106784,
    -0.002776865553098337,
    -0.002097208896757230,
    0.000000000000000016,
    0.002386188742962422,
    0.003595220170048398,
    0.002706428870995317,
    -0.000000000000000005,
    -0.003061312463468430,
    -0.004600428137275278,
    -0.003454941587499600,
    0.000000000000000006,
    0.003892353254050469,
    0.005839821854020543,
    0.004379814063152532,
    -0.000000000000000006,
    -0.004925228845166435,
    -0.007386047955257199,
    -0.005538718245278741,
    0.000000000000000007,
    0.006233452760818158,
    0.009357184582996936,
    0.007026886748910988,
    -0.000000000000000008,
    -0.007942635762930679,
    -0.011958781973950710,
    -0.009013425995813930,
    0.000000000000000008,
    0.010285905689420328,
    0.015581905289844846,
    0.011828851772232573,
    -0.000000000000000009,
    -0.013747940475479489,
    -0.021069155077820808,
    -0.016214150688088610,
    0.000000000000000009,
    0.019523150657378689,
    0.030618965552113523,
    0.024233864183139785,
    -0.000000000000000010,
    -0.031544575099683642,
    -0.052306756161031615,
    -0.044571677018128447,
    0.000000000000000010,
    0.074743412547564850,
    0.158858746421546015,
    0.224918453901548160,
    0.249917261731182699,
    0.224918453901548160,
    0.158858746421546015,
    0.074743412547564850,
    0.000000000000000010,
    -0.044571677018128447,
    -0.052306756161031615,
    -0.031544575099683642,
    -0.000000000000000010,
    0.024233864183139785,
    0.030618965552113526,
    0.019523150657378693,
    0.000000000000000009,
    -0.016214150688088614,
    -0.021069155077820808,
    -0.013747940475479491,
    -0.000000000000000009,
    0.011828851772232573,
    0.015581905289844844,
    0.010285905689420330,
    0.000000000000000008,
    -0.009013425995813933,
    -0.011958781973950712,
    -0.007942635762930679,
    -0.000000000000000008,
    0.007026886748910989,
    0.009357184582996938,
    0.006233452760818156,
    0.000000000000000007,
    -0.005538718245278742,
    -0.007386047955257199,
    -0.004925228845166436,
    -0.000000000000000006,
    0.004379814063152534,
    0.005839821854020542,
    0.003892353254050469,
    0.000000000000000006,
    -0.003454941587499600,
    -0.004600428137275276,
    -0.003061312463468430,
    -0.000000000000000005,
    0.002706428870995317,
    0.003595220170048400,
    0.002386188742962423,
    0.000000000000000016,
    -0.002097208896757230,
    -0.002776865553098338,
    -0.001836664205106785,
    -0.000000000000000004,
    0.001602166267302542,
    0.002112884021067343,
    0.001391670785141685,
    -0.000000000000000004,
    -0.001203406105977534,
    -0.001579842403752117,
    -0.001035817586776309,
    -0.000000000000000002,
    0.000887523932257827,
    0.001159965940927728,
    0.000757282701751800,
    0.000000000000000005,
    -0.000643962913379392,
    -0.000839081381695143,
    -0.000546523198228379,
    -0.000000000000000001,
    0.000463994335931762,
    0.000605332650800241,
    0.000395465279877511,
    -0.000000000000000001,
    -0.000340071984727431,
    -0.000448364890214722,
    -0.000296988501407617,
    -0.000000000000000001,
    0.000265419919503219,
    0.000358803521938844,
    0.000244596824120296,
    0.000000000000000002,
    -0.000233771000961963,
};

// https://fiiir.com/
// Gauss L=32
static const complex float bb_fir_tx_td[SCF_FIR_LEN_TX] = {
    0.002855073321371961,
    0.004217810045954073,
    0.006070979752995526,
    0.008513978883235714,
    0.011633445764465426,
    0.015487673582478250,
    0.020089356277603183,
    0.025389133050128021,
    0.031263078326193357,
    0.037507457211444148,
    0.043843530949484712,
    0.049933894562348549,
    0.055409900908815000,
    0.059907517556544401,
    0.063106964465950877,
    0.064770205340986750,
    0.064770205340986750,
    0.063106964465950877,
    0.059907517556544401,
    0.055409900908815000,
    0.049933894562348549,
    0.043843530949484712,
    0.037507457211444148,
    0.031263078326193357,
    0.025389133050128021,
    0.020089356277603183,
    0.015487673582478250,
    0.011633445764465426,
    0.008513978883235714,
    0.006070979752995526,
    0.004217810045954073,
    0.002855073321371961,
};
