#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <math.h>
#include <stdlib.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <time.h>
#include <assert.h>

#include "scf.h"
#include "scf_filter.h"
#include "scf_code.h"
#include "scf_tx.h"
#include "scf_rx.h"
#include "scf_fec.h"

#define RX_SIGNAL_LEN ((SCF_PKT_LEN + 2 + SCF_RX_LATENCY) * SCF_SYMBOL_LEN) /* samples */
#define TX_SIGNAL_LEN (SCF_PKT_LEN * SCF_SYMBOL_LEN) /* samples */
#define CARRIER_FREQ 1900.0f /* Hz*/

gsl_rng *rng;

uint8_t tx_message[SCF_MSG_LEN];
uint8_t tx_packet[SCF_PKT_LEN];
float *tx_signal;
float *noise;
float *rx_signal;
struct scf_soft_symbol rx_packet[SCF_PKT_LEN + 1];
uint8_t rx_message[SCF_MSG_LEN];

uint64_t get_msec(void)
{
    struct timespec ts;
    clock_gettime(CLOCK_REALTIME, &ts);

    return ts.tv_sec * 1000 + ts.tv_nsec / 1000000;
}

void dump_signal(char *filename, float *signal, size_t len)
{
    int f = open(filename, O_CREAT | O_TRUNC | O_RDWR, S_IRUSR | S_IWUSR);
    if (f < 0) {
        return;
    }

    for (size_t i = 0; i < len; i++) {
        int rv = write(f, &signal[i], sizeof(signal[i]));
        assert(rv > 0);
    }

    close(f);
}

void generate_message(uint8_t *msg)
{
    for (size_t i = 0; i < SCF_MSG_LEN; i++) {
        msg[i] = gsl_rng_get(rng) % SCF_SYMBOL_M;
    }
}

void generate_noise(float *out, size_t len, float sigma)
{
    for (size_t i = 0; i < len; i++) {
        out[i] = gsl_ran_gaussian(rng, sigma);
    }
}

void add_cw_interferer(float *out, size_t len, float freq, float amplitude)
{
    float phase = 0.0f;

    for (size_t i = 0; i < len; i++) {
        out[i] += amplitude * sinf(phase);
        phase += 2.0f * M_PI * freq * (1.0f / (float) SCF_SRATE);
        if (phase > 2.0f * M_PI) {
            phase -= 2.0f * M_PI;
        }
    }
}

void add_signal(float *out, float *in1, float *in2, size_t len)
{
    for (size_t i = 0; i < len; i++) {
        out[i] = in1[i] + in2[i];
    }
}

float measure_power(float *signal, size_t len)
{
    double energy = 0.0f;

    for (size_t i = 0; i < len; i++) {
        energy += signal[i] * signal[i];
    }

    return energy / (double) len;
}

bool run_test(void)
{
    scf_code_generate(gsl_rng_get(rng));

    // TX

    generate_message(tx_message);
    scf_fec_encode(tx_packet, tx_message);

    size_t tx_rand_delay = gsl_rng_uniform_int(rng, SCF_SYMBOL_LEN);
    float tx_rand_freq_offset = (gsl_rng_uniform(rng) - 0.5f) * 2.0f * ((float) SCF_SRATE / (float) SCF_CHIP_LEN);

    scf_tx_init(CARRIER_FREQ + tx_rand_freq_offset);
    for (size_t i = 0; i < SCF_PKT_LEN; i++) {
        size_t signal_i = tx_rand_delay + i * SCF_SYMBOL_LEN;
        scf_tx(&tx_signal[signal_i], tx_packet[i]);
    }
    //dump_signal("dump_tx_signal.raw", tx_signal, SIGNAL_LEN);

    for (size_t i = 0; i < RX_SIGNAL_LEN - 100; i++) {
        // Multipath
        //tx_signal[i] += 1.0f * tx_signal[i + 16];
        //tx_signal[i] *= 0.65f;
    }

    float signal_power = measure_power(&tx_signal[tx_rand_delay], TX_SIGNAL_LEN);

    // Channel

    generate_noise(noise, RX_SIGNAL_LEN, 8.0f);
    add_cw_interferer(noise, RX_SIGNAL_LEN, CARRIER_FREQ, 0.0001f);
    float noise_power = measure_power(&noise[tx_rand_delay], TX_SIGNAL_LEN);

    float snr = 10.0f * log10f(signal_power / noise_power);
    float user_bitrate = ((float) SCF_MSG_LEN / (float) SCF_PKT_LEN /* code ratio */) * (logf(SCF_SYMBOL_M) / logf(2.0f)) * (float) SCF_SRATE / (float) SCF_SYMBOL_LEN;
    float channel_bandwidth = (float) SCF_SRATE / 2.0f;
    float ebno = snr - 10.0f * log10f(user_bitrate / channel_bandwidth);
    printf(" snr: %f dB\n", snr);
    printf("ebno: %f dB\n", ebno);

    add_signal(rx_signal, tx_signal, noise, RX_SIGNAL_LEN);
    //dump_signal("dump_rx_signal.raw", rx_signal, SIGNAL_LEN);

    // RX

    scf_rx_init(CARRIER_FREQ);
    unsigned int symbol_count = 0;
    bool message_is_decoded = false;
    uint64_t decoder_time_start = get_msec();

    for (
        size_t i = 0;
        i < RX_SIGNAL_LEN - SCF_CHIP_LEN + 1;
        i += SCF_CHIP_LEN
    ) {
        bool have_decode = scf_rx_chip(&rx_packet[SCF_PKT_LEN], &rx_signal[i]);
        if (have_decode) {
            symbol_count++;
            memmove(&rx_packet[0], &rx_packet[1], sizeof(rx_packet) - sizeof(rx_packet[0]));
            if (scf_fec_decode(rx_message, rx_packet)) {
                assert(!memcmp(tx_message, rx_message, SCF_MSG_LEN));
                message_is_decoded = true;
                break;
            }
        }
    }

    uint64_t decoder_time = get_msec() - decoder_time_start;
    printf("%s received after %u symbols in %lu msec\n\n", message_is_decoded ? "    " : " NOT", symbol_count, decoder_time);

    return message_is_decoded;
}

int main(int argc, char **argv)
{
    tx_signal = calloc(RX_SIGNAL_LEN, sizeof(tx_signal[0]));
    noise = calloc(RX_SIGNAL_LEN, sizeof(noise[0]));
    rx_signal = calloc(RX_SIGNAL_LEN, sizeof(rx_signal[0]));

    assert(tx_signal);
    assert(noise);
    assert(rx_signal);

    rng = gsl_rng_alloc(gsl_rng_taus2);
    if (!rng) {
        perror("RNG initialization error\n");
        exit(1);
    }
    gsl_rng_set(rng, time(NULL));
    scf_fec_init();

    unsigned int decoded_message_cnt = 0;
    const unsigned int test_cnt = 100;
    for (unsigned int i = 0; i < test_cnt; i++) {
        printf("Test %u\n", i);
        if (run_test()) {
            decoded_message_cnt++;
        }
    }
    float reception_probability = (float) decoded_message_cnt / (float) test_cnt;
    printf("Packet reception probability %f\n", reception_probability);
}
